<?php
/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

require 'vendor/autoload.php';

use PrestaShop\Module\PsAccounts\Service\PsAccountsService;
use PrestaShopCorp\Billing\Services\BillingService;

class Dropinblog extends Module
{
    public function __construct()
    {
        $this->name = 'dropinblog';
        $this->tab = 'front_office_features';
        $this->version = '1.0.2';
        $this->author = 'DropInBlog';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = [
            'min' => '1.7.0',
            'max' => '1.7.8.8',
        ];
        $this->bootstrap = true;
        $this->module_key = '4fec6395a49b5101ba98ca75483cd80f';

        parent::__construct();

        $this->displayName = $this->l('DropInBlog');
        $this->description = $this->l('DropInBlog is a powerful and easy-to-use blogging module that allows you to add a full-featured blog to your Prestashop store.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');
    }

    public function install(): bool
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        return
            parent::install()
            && $this->getService('ps_accounts.installer')->install()
            && $this->registerHook('moduleRoutes')
            && $this->registerHook('actionFrontControllerSetMedia')
            && $this->registerHook('displayHome')
            && Configuration::updateValue('DROPINBLOG_DIRECTORY', 'blog')
            && Configuration::updateValue('DROPINBLOG_CDN_URL', 'https://io.dropinblog.com')
            && Configuration::updateValue('DROPINBLOG_API_URL', 'https://api.dropinblog.com/v1/prestashop')
            && Configuration::updateValue('DROPINBLOG_WEBHOOK_URL', 'https://api.dropinblog.com/webhooks/prestashop');
    }

    public function uninstall(): bool
    {
        return
            parent::uninstall()
            && Configuration::deleteByName('DROPINBLOG_ID')
            && Configuration::deleteByName('DROPINBLOG_LINKED')
            && Configuration::deleteByName('DROPINBLOG_DIRECTORY')
            && Configuration::deleteByName('DROPINBLOG_CDN_URL')
            && Configuration::deleteByName('DROPINBLOG_API_URL')
            && Configuration::deleteByName('DROPINBLOG_WEBHOOK_URL')
            && Configuration::deleteByName('DROPINBLOG_LOGIN_TOKEN')
            && Configuration::deleteByName('DROPINBLOG_RECENT_POSTS_TITLE')
            && Configuration::deleteByName('DROPINBLOG_RECENT_POSTS_COUNT')
            && Configuration::deleteByName('DROPINBLOG_RECENT_POSTS_CATEGORY_FILTER');
    }

    public function getService($serviceName)
    {
        $service = new \PrestaShop\ModuleLibServiceContainer\DependencyInjection\ServiceContainer(
            $this->name,
            $this->getLocalPath()
        );

        return $service->getService($serviceName);
    }

    public function hookActionFrontControllerSetMedia()
    {
        $this->context->controller->registerJavascript($this->name . '_proxy', Configuration::get('DROPINBLOG_CDN_URL') . '/prestajs/' . Configuration::get('DROPINBLOG_ID') . '.js', [
            'server' => 'remote',
            'position' => 'bottom',
            'priority' => 999,
        ]);
        $this->context->controller->registerStylesheet($this->name . '_css', Configuration::get('DROPINBLOG_CDN_URL') . '/css/templates.css', [
            'server' => 'remote',
            'position' => 'bottom',
            'priority' => 999,
        ]);
    }

    public function hookDisplayHome($params)
    {
        // api params
        $apiParams = [
            'b' => Configuration::get('DROPINBLOG_ID'),
            'recentposts' => Configuration::get('DROPINBLOG_RECENT_POSTS_COUNT') ?: 2,
            'recentpostscategories' => Configuration::get('DROPINBLOG_RECENT_POSTS_CATEGORY_FILTER'),
            'format' => 'json',
        ];

        // get the blog data
        $guzzleClient = new \GuzzleHttp\Client();
        $response = $guzzleClient->get(Configuration::get('DROPINBLOG_API_URL') . '?' . http_build_query($apiParams));
        $blogResponse = json_decode($response->getBody());
        $blogData = $blogResponse->data;

        if ($blogResponse->status != 'failed') {
            // assign data to smarty
            $this->context->smarty->assign([
                'blogData' => $blogData,
                'title' => Configuration::get('DROPINBLOG_RECENT_POSTS_TITLE') ?: 'Recent Posts',
            ]);

            return $this->display(__FILE__, 'dropinblog.tpl');
        }
    }

    public function getContent()
    {
        // allow to auto-install Account
        $accountsInstaller = $this->getService('ps_accounts.installer');
        $accountsInstaller->install();

        try {
            // account service
            $accountsFacade = $this->getService('ps_accounts.facade');
            $accountsService = $accountsFacade->getPsAccountsService();
            Media::addJsDef(['contextPsAccounts' => $accountsFacade->getPsAccountsPresenter()->present($this->name)]);

            // billing service
            $billingService = $this->getService('ps_billings.service');

            // handle settings save
            if (Tools::isSubmit('saveSettings')) {
                $alertOutput = $this->saveSettings($accountsService);
            }

            // check if account is linked
            if ($accountsService->isAccountLinked() && $this->hasSubscription($billingService) && !Configuration::get('DROPINBLOG_LINKED')) {
                $this->register($accountsService, $billingService);
            }

            // billing presenter
            $billingFacade = $this->getService('ps_billings.facade');
            Media::addJsDef($billingFacade->present([
                'logo' => $this->getLocalPath() . 'views/img/logo.png',
                'tosLink' => 'https://dropinblog.com/privacy/',
                'privacyLink' => 'https://dropinblog.com/privacy/',
                'emailSupport' => 'info@dropinblog.com',
            ]));

            // smarty assignments
            $this->context->smarty->assign('urlAccountsCdn', $accountsService->getAccountsVueCdn());
            $this->context->smarty->assign('pathVendor', $this->getPathUri() . 'views/js/chunk-vendors.js');
            $this->context->smarty->assign('pathApp', $this->getPathUri() . 'views/js/app.js');
            $this->context->smarty->assign('baseUrl', Context::getContext()->shop->getBaseURL(true));
            $this->context->smarty->assign('blogDirectory', Configuration::get('DROPINBLOG_DIRECTORY'));
            $this->context->smarty->assign('recentPostsCount', Configuration::get('DROPINBLOG_RECENT_POSTS_COUNT') ?: 2);
            $this->context->smarty->assign('recentPostsTitle', Configuration::get('DROPINBLOG_RECENT_POSTS_TITLE') ?: 'Recent Posts');
            $this->context->smarty->assign('recentPostsCategoryFilter', Configuration::get('DROPINBLOG_RECENT_POSTS_CATEGORY_FILTER'));
            $this->context->smarty->assign('alertOutput', $alertOutput ?? '');
            $this->context->smarty->assign('showSettings', $accountsService->isAccountLinked() && $this->hasSubscription($billingService));
            $this->context->smarty->assign('loginUrl', 'https://app.dropinblog.com/login/integration/' . Configuration::get('DROPINBLOG_LOGIN_TOKEN'));
        } catch (Exception $e) {
            $this->context->controller->errors[] = $e->getMessage();
            return '';
        }

        // return view
        return $this->context->smarty->fetch($this->getLocalPath() . 'views/templates/admin/configure.tpl');
    }

    protected function hasSubscription(BillingService $billingService): bool
    {
        $currentCustomer = $billingService->getCurrentCustomer();
        $currentSubscription = $billingService->getCurrentSubscription();

        if ($currentCustomer['success'] && isset($currentCustomer['body']['primary_payment_source_id']) && isset($currentSubscription['httpStatus']) && $currentSubscription['httpStatus'] == 200 && !in_array($currentSubscription['body']['status'], ['cancelled'])) {
            return true;
        }
        return false;
    }

    protected function register(PsAccountsService $accountService, BillingService $billingService): void
    {
        // send webhook
        $guzzleClient = new \GuzzleHttp\Client();
        $response = $guzzleClient->post(Configuration::get('DROPINBLOG_WEBHOOK_URL'), [
            'json' => [
                'eventType' => 'register',
                'data' => [
                    'shopId' => $accountService->getShopUuid(),
                    'shopUrl' => Context::getContext()->shop->getBaseURL(true),
                    'domain' => Configuration::get('PS_SHOP_DOMAIN'),
                    'directory' => Configuration::get('DROPINBLOG_DIRECTORY'),
                    'customer' => $billingService->getCurrentCustomer()['body'],
                    'subscription' => $billingService->getCurrentSubscription()['body'],
                ],
            ],
        ]);
        $responseData = $response->json();

        // update config
        Configuration::updateValue('DROPINBLOG_ID', $responseData['refId']);
        Configuration::updateValue('DROPINBLOG_LOGIN_TOKEN', $responseData['loginToken']);
        Configuration::updateValue('DROPINBLOG_LINKED', true);
    }

    protected function saveSettings(PsAccountsService $accountService): ?string
    {
        // retrieve the value set by the user
        $directory = (string) Tools::getValue('directory');

        if (empty($directory) || !Validate::isGenericName($directory)) {
            // invalid value, return an error message
            return $this->displayError($this->l('You must set a blog url'));
        } else {
            if ($directory !== Configuration::get('DROPINBLOG_DIRECTORY')) {
                // send webhook
                $guzzleClient = new \GuzzleHttp\Client();
                $guzzleClient->post(Configuration::get('DROPINBLOG_WEBHOOK_URL'), [
                    'json' => [
                        'eventType' => 'directorychanged',
                        'data' => [
                            'shopId' => $accountService->getShopUuid(),
                            'shopUrl' => Context::getContext()->shop->getBaseURL(true),
                            'domain' => Configuration::get('PS_SHOP_DOMAIN'),
                            'directory' => $directory,
                        ],
                    ],
                ]);
            }

            // value is ok, update it and display a confirmation message
            Configuration::updateValue('DROPINBLOG_DIRECTORY', $directory);
            Configuration::updateValue('DROPINBLOG_RECENT_POSTS_COUNT', Tools::getValue('recentPostsCount'));
            Configuration::updateValue('DROPINBLOG_RECENT_POSTS_TITLE', Tools::getValue('recentPostsTitle'));
            Configuration::updateValue('DROPINBLOG_RECENT_POSTS_CATEGORY_FILTER', Tools::getValue('recentPostsCategoryFilter'));
            return $this->displayConfirmation($this->l('Settings updated'));
        }
    }

    public function hookModuleRoutes($params): array
    {
        $directory = Configuration::get('DROPINBLOG_DIRECTORY');

        $my_routes = [
            $this->name . '_list' => [
                'controller' => 'list',
                'rule' => $directory,
                'keywords' => [],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_list_page' => [
                'controller' => 'list',
                'rule' => $directory . '/page/{page}',
                'keywords' => [
                    'page' => [
                        'regexp' => '[0-9]+',
                        'param' => 'page',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_list_category' => [
                'controller' => 'list',
                'rule' => $directory . '/category/{categoryslug}',
                'keywords' => [
                    'categoryslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'categoryslug',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_list_category_page' => [
                'controller' => 'list',
                'rule' => $directory . '/category/{categoryslug}/page/{page}',
                'keywords' => [
                    'categoryslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'categoryslug',
                    ],
                    'page' => [
                        'regexp' => '[0-9]+',
                        'param' => 'page',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_list_author' => [
                'controller' => 'list',
                'rule' => $directory . '/author/{authorslug}',
                'keywords' => [
                    'authorslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'authorslug',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_list_author_page' => [
                'controller' => 'list',
                'rule' => $directory . '/author/{authorslug}/page/{page}',
                'keywords' => [
                    'authorslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'authorslug',
                    ],
                    'page' => [
                        'regexp' => '[0-9]+',
                        'param' => 'page',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_feed' => [
                'controller' => 'feed',
                'rule' => $directory . '/feed',
                'keywords' => [],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_feed_category' => [
                'controller' => 'feed',
                'rule' => $directory . '/feed/category/{categoryslug}',
                'keywords' => [
                    'categoryslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'categoryslug',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_feed_author' => [
                'controller' => 'feed',
                'rule' => $directory . '/feed/author/{authorslug}',
                'keywords' => [
                    'authorslug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'authorslug',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_sitemap' => [
                'controller' => 'sitemap',
                'rule' => $directory . '/sitemap.xml',
                'keywords' => [],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
            $this->name . '_post' => [
                'controller' => 'single',
                'rule' => $directory . '/{slug}',
                'keywords' => [
                    'slug' => [
                        'regexp' => '[_a-zA-Z0-9-\pL]*',
                        'param' => 'slug',
                    ],
                ],
                'params' => [
                    'fc' => 'module',
                    'module' => $this->name,
                ],
            ],
        ];

        return $my_routes;
    }
}
