<?php
/**
 * Copyright © Lyra Network.
 * This file is part of EpayNC plugin for PrestaShop. See COPYING.md for license details.
 *
 * @author    Lyra Network (https://www.lyra.com/)
 * @copyright Lyra Network
 * @license   https://opensource.org/licenses/afl-3.0.php Academic Free License (AFL 3.0)
 */

if (! defined('_PS_VERSION_')) {
    exit;
}

/**
 * Class that renders payment module administration interface.
 */
class EpayncHelperForm
{
    private function __construct()
    {
        // Do not instantiate this class.
    }

    public static function getAdminFormContext()
    {
        $context = Context::getContext();

        /* @var Epaync */
        $epaync = Module::getInstanceByName('epaync');

        $languages = array();
        foreach (EpayncApi::getSupportedLanguages() as $code => $label) {
            $languages[$code] = $epaync->l($label, 'epaynchelperform');
        }

        asort($languages);

        $category_options = array(
            'FOOD_AND_GROCERY' => $epaync->l('Food and grocery', 'epaynchelperform'),
            'AUTOMOTIVE' => $epaync->l('Automotive', 'epaynchelperform'),
            'ENTERTAINMENT' => $epaync->l('Entertainment', 'epaynchelperform'),
            'HOME_AND_GARDEN' => $epaync->l('Home and garden', 'epaynchelperform'),
            'HOME_APPLIANCE' => $epaync->l('Home appliance', 'epaynchelperform'),
            'AUCTION_AND_GROUP_BUYING' => $epaync->l('Auction and group buying', 'epaynchelperform'),
            'FLOWERS_AND_GIFTS' => $epaync->l('Flowers and gifts', 'epaynchelperform'),
            'COMPUTER_AND_SOFTWARE' => $epaync->l('Computer and software', 'epaynchelperform'),
            'HEALTH_AND_BEAUTY' => $epaync->l('Health and beauty', 'epaynchelperform'),
            'SERVICE_FOR_INDIVIDUAL' => $epaync->l('Service for individual', 'epaynchelperform'),
            'SERVICE_FOR_BUSINESS' => $epaync->l('Service for business', 'epaynchelperform'),
            'SPORTS' => $epaync->l('Sports', 'epaynchelperform'),
            'CLOTHING_AND_ACCESSORIES' => $epaync->l('Clothing and accessories', 'epaynchelperform'),
            'TRAVEL' => $epaync->l('Travel', 'epaynchelperform'),
            'HOME_AUDIO_PHOTO_VIDEO' => $epaync->l('Home audio, photo, video', 'epaynchelperform'),
            'TELEPHONY' => $epaync->l('Telephony', 'epaynchelperform')
        );

        // Get documentation links.
        $doc_files = array();
        $filenames = glob(_PS_MODULE_DIR_ . 'epaync/installation_doc/' . EpayncTools::getDocPattern());

        $doc_languages = array(
            'fr' => 'Français',
            'en' => 'English',
            'es' => 'Español',
            'de' => 'Deutsch'
            // Complete when other languages are managed.
        );

        foreach ($filenames as $filename) {
            $base_filename = basename($filename, '.pdf');
            $lang = Tools::substr($base_filename, -2); // Extract language code.

            $doc_files[$base_filename . '.pdf'] = $doc_languages[$lang];
        }

        $placeholders = self::getArrayConfig('EPAYNC_STD_REST_PLACEHLDR');
        if (empty($placeholders)) {
            $placeholders = array('pan' => '', 'expiry' => '', 'cvv' => '');
        }

        $enabledCountries = Country::getCountries((int) $context->language->id, true);
        $all_countries = Country::getCountries((int) $context->language->id, false);
        $countryList = array();
        foreach ($enabledCountries as $value) {
            $countryList['ps_countries'][$value['iso_code']] = $value['name'];
        }

        foreach (EpayncTools::$submodules as $key => $module) {
            $module_class_name = 'Epaync' . $module.'Payment';
            $instance_module = new $module_class_name();
            if (method_exists($instance_module, 'getCountries') && $instance_module->getCountries()) {
                $submodule_specific_countries = $instance_module->getCountries();
                foreach ($submodule_specific_countries as $country) {
                    if (isset($countryList['ps_countries'][$country])) {
                        $countryList[$key][$country] = $countryList['ps_countries'][$country];
                    }
                }
            }
        }

        foreach ($all_countries as $value) {
            if ($value['iso_code'] === 'FR') {
                $countryList['FULLCB']['FR'] = $value['name'];
                break;
            }
        }

        $tpl_vars = array(
            'epaync_support_email' => EpayncTools::getDefault('SUPPORT_EMAIL'),
            'epaync_formatted_support_email' => EpayncApi::formatSupportEmails(EpayncTools::getDefault('SUPPORT_EMAIL')),
            'epaync_plugin_version' => EpayncTools::getDefault('PLUGIN_VERSION'),
            'epaync_gateway_version' => EpayncTools::getDefault('GATEWAY_VERSION'),
            'epaync_contrib' => EpayncTools::getContrib(),
            'epaync_installed_modules' => EpayncTools::getModulesInstalled(),
            'epaync_card_data_entry_modes' => EpayncTools::getCardDataEntryModes(),
            'epaync_employee' => $context->employee,

            'epaync_plugin_features' => EpayncTools::$plugin_features,
            'epaync_request_uri' => $_SERVER['REQUEST_URI'],

            'epaync_doc_files' => $doc_files,
            'epaync_enable_disable_options' => array(
                'False' => $epaync->l('Disabled', 'epaynchelperform'),
                'True' => $epaync->l('Enabled', 'epaynchelperform')
            ),
            'epaync_mode_options' => array(
                'TEST' => $epaync->l('TEST', 'epaynchelperform'),
                'PRODUCTION' => $epaync->l('PRODUCTION', 'epaynchelperform')
            ),
            'epaync_language_options' => $languages,
            'epaync_validation_mode_options' => array(
                '' => $epaync->l('Bank Back Office configuration', 'epaynchelperform'),
                '0' => $epaync->l('Automatic', 'epaynchelperform'),
                '1' => $epaync->l('Manual', 'epaynchelperform')
            ),
            'epaync_payment_cards_options' => array('' => $epaync->l('ALL', 'epaynchelperform')) + EpayncTools::getSupportedCardTypes(),
            'epaync_multi_payment_cards_options' => array('' => $epaync->l('ALL', 'epaynchelperform')) + EpayncTools::getSupportedMultiCardTypes(),
            'epaync_category_options' => $category_options,
            'epaync_yes_no_options' => array(
                'False' => $epaync->l('No', 'epaynchelperform'),
                'True' => $epaync->l('Yes', 'epaynchelperform')
            ),
            'epaync_delivery_type_options' => array(
                'PACKAGE_DELIVERY_COMPANY' => $epaync->l('Delivery company', 'epaynchelperform'),
                'RECLAIM_IN_SHOP' => $epaync->l('Reclaim in shop', 'epaynchelperform'),
                'RELAY_POINT' => $epaync->l('Relay point', 'epaynchelperform'),
                'RECLAIM_IN_STATION' => $epaync->l('Reclaim in station', 'epaynchelperform')
            ),
            'epaync_delivery_speed_options' => array(
                'STANDARD' => $epaync->l('Standard', 'epaynchelperform'),
                'EXPRESS' => $epaync->l('Express', 'epaynchelperform'),
                'PRIORITY' => $epaync->l('Priority', 'epaynchelperform')
            ),
            'epaync_delivery_delay_options' => array(
                'INFERIOR_EQUALS' => $epaync->l('<= 1 hour', 'epaynchelperform'),
                'SUPERIOR' => $epaync->l('> 1 hour', 'epaynchelperform'),
                'IMMEDIATE' => $epaync->l('Immediate', 'epaynchelperform'),
                'ALWAYS' => $epaync->l('24/7', 'epaynchelperform')
            ),
            'epaync_failure_management_options' => array(
                EpayncTools::ON_FAILURE_RETRY => $epaync->l('Go back to checkout', 'epaynchelperform'),
                EpayncTools::ON_FAILURE_SAVE => $epaync->l('Save order and go back to order history', 'epaynchelperform')
            ),
            'epaync_cart_management_options' => array(
                EpayncTools::EMPTY_CART => $epaync->l('Empty cart to avoid amount errors', 'epaynchelperform'),
                EpayncTools::KEEP_CART => $epaync->l('Keep cart (PrestaShop default behavior)', 'epaynchelperform')
            ),
            'epaync_card_data_mode_options' => array(
                '1' => $epaync->l('Bank data acquisition on payment gateway', 'epaynchelperform'),
                '2' => $epaync->l('Card type selection on merchant site', 'epaynchelperform'),
                '4' => $epaync->l('Payment page integrated to checkout process (iframe mode)', 'epaynchelperform'),
                '5' => $epaync->l('Embedded payment fields on merchant site (REST API)', 'epaynchelperform'),
                '6' => $epaync->l('Embedded payment fields in a pop-in (REST API)', 'epaynchelperform')
            ),
            'epaync_countries_options' => array(
                '1' => $epaync->l('All Allowed Countries', 'epaynchelperform'),
                '2' => $epaync->l('Specific Countries', 'epaynchelperform')
            ),
            'epaync_countries_list' => $countryList,
            'epaync_card_selection_mode_options' => array(
                '1' => $epaync->l('On payment gateway', 'epaynchelperform'),
                '2' => $epaync->l('On merchant site', 'epaynchelperform')
            ),
            'epaync_default_multi_option' => array(
                'label' => '',
                'min_amount' => '',
                'max_amount' => '',
                'contract' => '',
                'count' => '',
                'period' => '',
                'first' => ''
            ),
            'epaync_default_oney_option' => array(
                'label' => '',
                'code' => '',
                'min_amount' => '',
                'max_amount' => '',
                'count' => '',
                'rate' => ''
            ),
            'epaync_default_franfinance_option' => array(
                'label' => '',
                'count' => '3',
                'fees' => '-1',
                'min_amount' => '100',
                'max_amount' => '3000'
            ),
            'franfinance_count' => array(
                '3' => '3x',
                '4' => '4x'
            ),
            'fees_options' => array(
                '-1' => $epaync->l('Bank Back Office configuration', 'epaynchelperform'),
                '0' => $epaync->l('Without fees', 'epaynchelperform'),
                '1' => $epaync->l('With fees', 'epaynchelperform')
            ),
            'epaync_default_other_payment_means_option' => array(
                'title' => '',
                'code' => '',
                'min_amount' => '',
                'max_amount' => '',
                'validation' => '-1',
                'capture' => '',
                'cart' => 'False'
            ),
            'epaync_default_extra_payment_means_option' => array(
                'code' => '',
                'title' => ''
            ),
            'epaync_std_rest_theme_options' => array(
                'classic' =>  'Classic',
                'material' => 'Material'
            ),

            'prestashop_categories' => Category::getCategories((int) $context->language->id, true, false),
            'prestashop_languages' => Language::getLanguages(false),
            'prestashop_lang' => Language::getLanguage((int) $context->language->id),
            'prestashop_carriers' => Carrier::getCarriers(
                (int) $context->language->id,
                true,
                false,
                false,
                null,
                Carrier::ALL_CARRIERS
            ),
            'prestashop_groups' => self::getAuthorizedGroups(),
            'epaync_sepa_mandate_mode_options' => array(
                'PAYMENT' => $epaync->l('One-off SEPA direct debit', 'epaynchelperform'),
                'REGISTER_PAY' => $epaync->l('Register a recurrent SEPA mandate with direct debit', 'epaynchelperform'),
                'REGISTER' => $epaync->l('Register a recurrent SEPA mandate without direct debit', 'epaynchelperform')
            ),

            'EPAYNC_ENABLE_LOGS' => Configuration::get('EPAYNC_ENABLE_LOGS'),

            'EPAYNC_SITE_ID' => Configuration::get('EPAYNC_SITE_ID'),
            'EPAYNC_KEY_TEST' => Configuration::get('EPAYNC_KEY_TEST'),
            'EPAYNC_KEY_PROD' => Configuration::get('EPAYNC_KEY_PROD'),
            'EPAYNC_MODE' => Configuration::get('EPAYNC_MODE'),
            'EPAYNC_SIGN_ALGO' => Configuration::get('EPAYNC_SIGN_ALGO'),
            'EPAYNC_PLATFORM_URL' => Configuration::get('EPAYNC_PLATFORM_URL'),
            'EPAYNC_NOTIFY_URL' => self::getIpnUrl(),

            'EPAYNC_PUBKEY_TEST' => Configuration::get('EPAYNC_PUBKEY_TEST'),
            'EPAYNC_PRIVKEY_TEST' => Configuration::get('EPAYNC_PRIVKEY_TEST'),
            'EPAYNC_PUBKEY_PROD' => Configuration::get('EPAYNC_PUBKEY_PROD'),
            'EPAYNC_PRIVKEY_PROD' => Configuration::get('EPAYNC_PRIVKEY_PROD'),
            'EPAYNC_RETKEY_TEST' => Configuration::get('EPAYNC_RETKEY_TEST'),
            'EPAYNC_RETKEY_PROD' => Configuration::get('EPAYNC_RETKEY_PROD'),
            'EPAYNC_REST_NOTIFY_URL' => self::getIpnUrl(),
            'EPAYNC_REST_SERVER_URL' => Configuration::get('EPAYNC_REST_SERVER_URL'),
            'EPAYNC_REST_JS_CLIENT_URL' => Configuration::get('EPAYNC_REST_JS_CLIENT_URL'),

            'EPAYNC_DEFAULT_LANGUAGE' => Configuration::get('EPAYNC_DEFAULT_LANGUAGE'),
            'EPAYNC_AVAILABLE_LANGUAGES' => ! Configuration::get('EPAYNC_AVAILABLE_LANGUAGES') ?
                                            array('') :
                                            explode(';', Configuration::get('EPAYNC_AVAILABLE_LANGUAGES')),
            'EPAYNC_DELAY' => Configuration::get('EPAYNC_DELAY'),
            'EPAYNC_VALIDATION_MODE' => Configuration::get('EPAYNC_VALIDATION_MODE'),

            'EPAYNC_THEME_CONFIG' => self::getLangConfig('EPAYNC_THEME_CONFIG'),
            'EPAYNC_SHOP_NAME' => Configuration::get('EPAYNC_SHOP_NAME'),
            'EPAYNC_SHOP_URL' => Configuration::get('EPAYNC_SHOP_URL'),

            'EPAYNC_3DS_MIN_AMOUNT' => self::getArrayConfig('EPAYNC_3DS_MIN_AMOUNT'),

            'EPAYNC_REDIRECT_ENABLED' => Configuration::get('EPAYNC_REDIRECT_ENABLED'),
            'EPAYNC_REDIRECT_SUCCESS_T' => Configuration::get('EPAYNC_REDIRECT_SUCCESS_T'),
            'EPAYNC_REDIRECT_SUCCESS_M' => self::getLangConfig('EPAYNC_REDIRECT_SUCCESS_M'),
            'EPAYNC_REDIRECT_ERROR_T' => Configuration::get('EPAYNC_REDIRECT_ERROR_T'),
            'EPAYNC_REDIRECT_ERROR_M' => self::getLangConfig('EPAYNC_REDIRECT_ERROR_M'),
            'EPAYNC_RETURN_MODE' => Configuration::get('EPAYNC_RETURN_MODE'),
            'EPAYNC_FAILURE_MANAGEMENT' => Configuration::get('EPAYNC_FAILURE_MANAGEMENT'),
            'EPAYNC_CART_MANAGEMENT' => Configuration::get('EPAYNC_CART_MANAGEMENT'),

            'EPAYNC_SEND_CART_DETAIL' => Configuration::get('EPAYNC_SEND_CART_DETAIL'),
            'EPAYNC_COMMON_CATEGORY' => Configuration::get('EPAYNC_COMMON_CATEGORY'),
            'EPAYNC_CATEGORY_MAPPING' => self::getArrayConfig('EPAYNC_CATEGORY_MAPPING'),
            'EPAYNC_SEND_SHIP_DATA' => Configuration::get('EPAYNC_SEND_SHIP_DATA'),
            'EPAYNC_ONEY_SHIP_OPTIONS' => self::getArrayConfig('EPAYNC_ONEY_SHIP_OPTIONS'),

            'EPAYNC_STD_TITLE' => self::getLangConfig('EPAYNC_STD_TITLE'),
            'EPAYNC_STD_ENABLED' => Configuration::get('EPAYNC_STD_ENABLED'),
            'EPAYNC_STD_AMOUNTS' => self::getArrayConfig('EPAYNC_STD_AMOUNTS'),
            'EPAYNC_STD_DELAY' => Configuration::get('EPAYNC_STD_DELAY'),
            'EPAYNC_STD_VALIDATION' => Configuration::get('EPAYNC_STD_VALIDATION'),
            'EPAYNC_STD_PAYMENT_CARDS' => ! Configuration::get('EPAYNC_STD_PAYMENT_CARDS') ?
                                            array('') :
                                            explode(';', Configuration::get('EPAYNC_STD_PAYMENT_CARDS')),
            'EPAYNC_STD_PROPOSE_ONEY' => Configuration::get('EPAYNC_STD_PROPOSE_ONEY'),
            'EPAYNC_STD_CARD_DATA_MODE' => Configuration::get('EPAYNC_STD_CARD_DATA_MODE') ?
                                            Configuration::get('EPAYNC_STD_CARD_DATA_MODE') : '1',
            'EPAYNC_STD_REST_THEME' => Configuration::get('EPAYNC_STD_REST_THEME'),
            'EPAYNC_STD_REST_PLACEHLDR' => $placeholders,
            'EPAYNC_STD_REST_LBL_REGIST' => self::getLangConfig('EPAYNC_STD_REST_LBL_REGIST'),
            'EPAYNC_STD_REST_ATTEMPTS' => Configuration::get('EPAYNC_STD_REST_ATTEMPTS'),
            'EPAYNC_STD_1_CLICK_PAYMENT' => Configuration::get('EPAYNC_STD_1_CLICK_PAYMENT'),
            'EPAYNC_STD_CANCEL_IFRAME' => Configuration::get('EPAYNC_STD_CANCEL_IFRAME'),

            'EPAYNC_MULTI_TITLE' => self::getLangConfig('EPAYNC_MULTI_TITLE'),
            'EPAYNC_MULTI_ENABLED' => Configuration::get('EPAYNC_MULTI_ENABLED'),
            'EPAYNC_MULTI_AMOUNTS' => self::getArrayConfig('EPAYNC_MULTI_AMOUNTS'),
            'EPAYNC_MULTI_DELAY' => Configuration::get('EPAYNC_MULTI_DELAY'),
            'EPAYNC_MULTI_VALIDATION' => Configuration::get('EPAYNC_MULTI_VALIDATION'),
            'EPAYNC_MULTI_CARD_MODE' => Configuration::get('EPAYNC_MULTI_CARD_MODE'),
            'EPAYNC_MULTI_PAYMENT_CARDS' => ! Configuration::get('EPAYNC_MULTI_PAYMENT_CARDS') ?
                                            array('') :
                                            explode(';', Configuration::get('EPAYNC_MULTI_PAYMENT_CARDS')),
            'EPAYNC_MULTI_OPTIONS' => self::getArrayConfig('EPAYNC_MULTI_OPTIONS'),

            'EPAYNC_ANCV_TITLE' => self::getLangConfig('EPAYNC_ANCV_TITLE'),
            'EPAYNC_ANCV_ENABLED' => Configuration::get('EPAYNC_ANCV_ENABLED'),
            'EPAYNC_ANCV_AMOUNTS' => self::getArrayConfig('EPAYNC_ANCV_AMOUNTS'),
            'EPAYNC_ANCV_DELAY' => Configuration::get('EPAYNC_ANCV_DELAY'),
            'EPAYNC_ANCV_VALIDATION' => Configuration::get('EPAYNC_ANCV_VALIDATION'),

            'EPAYNC_ONEY_TITLE' => self::getLangConfig('EPAYNC_ONEY_TITLE'),
            'EPAYNC_ONEY_ENABLED' => Configuration::get('EPAYNC_ONEY_ENABLED'),
            'EPAYNC_ONEY_AMOUNTS' => self::getArrayConfig('EPAYNC_ONEY_AMOUNTS'),
            'EPAYNC_ONEY_DELAY' => Configuration::get('EPAYNC_ONEY_DELAY'),
            'EPAYNC_ONEY_VALIDATION' => Configuration::get('EPAYNC_ONEY_VALIDATION'),
            'EPAYNC_ONEY_ENABLE_OPTIONS' => Configuration::get('EPAYNC_ONEY_ENABLE_OPTIONS'),
            'EPAYNC_ONEY_OPTIONS' => self::getArrayConfig('EPAYNC_ONEY_OPTIONS'),

            'EPAYNC_ONEY34_TITLE' => self::getLangConfig('EPAYNC_ONEY34_TITLE'),
            'EPAYNC_ONEY34_ENABLED' => Configuration::get('EPAYNC_ONEY34_ENABLED'),
            'EPAYNC_ONEY34_AMOUNTS' => self::getArrayConfig('EPAYNC_ONEY34_AMOUNTS'),
            'EPAYNC_ONEY34_DELAY' => Configuration::get('EPAYNC_ONEY34_DELAY'),
            'EPAYNC_ONEY34_VALIDATION' => Configuration::get('EPAYNC_ONEY34_VALIDATION'),
            'EPAYNC_ONEY34_OPTIONS' => self::getArrayConfig('EPAYNC_ONEY34_OPTIONS'),

            'EPAYNC_FFIN_TITLE' => self::getLangConfig('EPAYNC_FFIN_TITLE'),
            'EPAYNC_FFIN_ENABLED' => Configuration::get('EPAYNC_FFIN_ENABLED'),
            'EPAYNC_FFIN_AMOUNTS' => self::getArrayConfig('EPAYNC_FFIN_AMOUNTS'),
            'EPAYNC_FFIN_OPTIONS' => self::getArrayConfig('EPAYNC_FFIN_OPTIONS'),

            'EPAYNC_FULLCB_TITLE' => self::getLangConfig('EPAYNC_FULLCB_TITLE'),
            'EPAYNC_FULLCB_ENABLED' => Configuration::get('EPAYNC_FULLCB_ENABLED'),
            'EPAYNC_FULLCB_AMOUNTS' => self::getArrayConfig('EPAYNC_FULLCB_AMOUNTS'),
            'EPAYNC_FULLCB_ENABLE_OPTS' => Configuration::get('EPAYNC_FULLCB_ENABLE_OPTS'),
            'EPAYNC_FULLCB_OPTIONS' => self::getArrayConfig('EPAYNC_FULLCB_OPTIONS'),

            'EPAYNC_SEPA_TITLE' => self::getLangConfig('EPAYNC_SEPA_TITLE'),
            'EPAYNC_SEPA_ENABLED' => Configuration::get('EPAYNC_SEPA_ENABLED'),
            'EPAYNC_SEPA_AMOUNTS' => self::getArrayConfig('EPAYNC_SEPA_AMOUNTS'),
            'EPAYNC_SEPA_DELAY' => Configuration::get('EPAYNC_SEPA_DELAY'),
            'EPAYNC_SEPA_VALIDATION' => Configuration::get('EPAYNC_SEPA_VALIDATION'),
            'EPAYNC_SEPA_MANDATE_MODE' => Configuration::get('EPAYNC_SEPA_MANDATE_MODE'),
            'EPAYNC_SEPA_1_CLICK_PAYMNT' => Configuration::get('EPAYNC_SEPA_1_CLICK_PAYMNT'),

            'EPAYNC_SOFORT_TITLE' => self::getLangConfig('EPAYNC_SOFORT_TITLE'),
            'EPAYNC_SOFORT_ENABLED' => Configuration::get('EPAYNC_SOFORT_ENABLED'),
            'EPAYNC_SOFORT_AMOUNTS' => self::getArrayConfig('EPAYNC_SOFORT_AMOUNTS'),

            'EPAYNC_PAYPAL_TITLE' => self::getLangConfig('EPAYNC_PAYPAL_TITLE'),
            'EPAYNC_PAYPAL_ENABLED' => Configuration::get('EPAYNC_PAYPAL_ENABLED'),
            'EPAYNC_PAYPAL_AMOUNTS' => self::getArrayConfig('EPAYNC_PAYPAL_AMOUNTS'),
            'EPAYNC_PAYPAL_DELAY' => Configuration::get('EPAYNC_PAYPAL_DELAY'),
            'EPAYNC_PAYPAL_VALIDATION' => Configuration::get('EPAYNC_PAYPAL_VALIDATION'),

            'EPAYNC_CHOOZEO_TITLE' => self::getLangConfig('EPAYNC_CHOOZEO_TITLE'),
            'EPAYNC_CHOOZEO_ENABLED' => Configuration::get('EPAYNC_CHOOZEO_ENABLED'),
            'EPAYNC_CHOOZEO_AMOUNTS' => self::getArrayConfig('EPAYNC_CHOOZEO_AMOUNTS'),
            'EPAYNC_CHOOZEO_DELAY' => Configuration::get('EPAYNC_CHOOZEO_DELAY'),
            'EPAYNC_CHOOZEO_OPTIONS' => self::getArrayConfig('EPAYNC_CHOOZEO_OPTIONS'),

            'EPAYNC_OTHER_GROUPED_VIEW' => Configuration::get('EPAYNC_OTHER_GROUPED_VIEW'),
            'EPAYNC_OTHER_ENABLED' => Configuration::get('EPAYNC_OTHER_ENABLED'),
            'EPAYNC_OTHER_TITLE' => self::getLangConfig('EPAYNC_OTHER_TITLE'),
            'EPAYNC_OTHER_AMOUNTS' => self::getArrayConfig('EPAYNC_OTHER_AMOUNTS'),
            'EPAYNC_OTHER_PAYMENT_MEANS' => self::getArrayConfig('EPAYNC_OTHER_PAYMENT_MEANS'),
            'EPAYNC_EXTRA_PAYMENT_MEANS' => self::getArrayConfig('EPAYNC_EXTRA_PAYMENT_MEANS')
        );

        foreach (EpayncTools::$submodules as $key => $module) {
            $tpl_vars['EPAYNC_' . $key . '_COUNTRY'] = Configuration::get('EPAYNC_' . $key . '_COUNTRY');
            $tpl_vars['EPAYNC_' . $key . '_COUNTRY_LST'] = ! Configuration::get('EPAYNC_' . $key . '_COUNTRY_LST') ?
                array() : explode(';', Configuration::get('EPAYNC_' . $key . '_COUNTRY_LST'));
        }

        if (! EpayncTools::$plugin_features['embedded']) {
            unset($tpl_vars['epaync_card_data_mode_options'][EpayncTools::MODE_EMBEDDED]);
            unset($tpl_vars['epaync_card_data_mode_options'][EpayncTools::MODE_POPIN]);
        }

        return $tpl_vars;
    }

    private static function getIpnUrl()
    {
        $shop = new Shop(Configuration::get('PS_SHOP_DEFAULT'));

        // SSL enabled on default shop?
        $id_shop_group = isset($shop->id_shop_group) ? $shop->id_shop_group : $shop->id_group_shop;
        $ssl = Configuration::get('PS_SSL_ENABLED', null, $id_shop_group, $shop->id);

        $ipn = ($ssl ? 'https://' . $shop->domain_ssl : 'http://' . $shop->domain)
            . $shop->getBaseURI() . 'modules/epaync/validation.php';

        return $ipn;
    }

    private static function getArrayConfig($name)
    {
        $value = @unserialize(Configuration::get($name));

        if (! is_array($value)) {
            $value = array();
        }

        return $value;
    }

    private static function getLangConfig($name)
    {
        $languages = Language::getLanguages(false);

        $result = array();
        foreach ($languages as $language) {
            $result[$language['id_lang']] = Configuration::get($name, $language['id_lang']);
        }

        return $result;
    }

    private static function getAuthorizedGroups()
    {
        $context = Context::getContext();

        /* @var Epaync */
        $epaync = Module::getInstanceByName('epaync');

        $sql = 'SELECT DISTINCT gl.`id_group`, gl.`name` FROM `' . _DB_PREFIX_ . 'group_lang` AS gl
            INNER JOIN `' . _DB_PREFIX_ . 'module_group` AS mg
            ON (
                gl.`id_group` = mg.`id_group`
                AND mg.`id_module` = ' . (int) $epaync->id . '
                AND mg.`id_shop` = ' . (int) $context->shop->id . '
            )
            WHERE gl.`id_lang` = ' . (int) $context->language->id;

        return Db::getInstance()->executeS($sql);
    }
}
